<#
.SYNOPSIS
  Looks in .\scripts\, finds the latest script by modified time, and executes it remotely via SSH.
#>
param(
  [Parameter(Mandatory=$true)][string]$RemoteIP
)

$RemoteUser = $env:REMOTE_USER
if (-not $RemoteUser) { $RemoteUser = "root" }
$RemotePort = if ($env:REMOTE_PORT) { $env:REMOTE_PORT } else { 22 }
$RemoteDir = if ($env:REMOTE_DIR) { $env:REMOTE_DIR } else { "/tmp" }
$Cleanup = if ($env:CLEANUP) { $env:CLEANUP } else { "yes" }
$TimeoutSecs = if ($env:TIMEOUT_SECS) { [int]$env:TIMEOUT_SECS } else { 900 }
$MaxScripts = 1
if ($env:MAX_SCRIPTS -and $env:MAX_SCRIPTS -match '^\d+$') {
  $parsed = [int]$env:MAX_SCRIPTS
  if ($parsed -gt 0) { $MaxScripts = $parsed }
}

function Get-OptionArray {
  param (
    [string]$Value,
    [string[]]$Default
  )
  if ($Value -and $Value.Trim()) {
    return $Value -split '\s+' | Where-Object { $_ }
  }
  return $Default
}

function Get-SafeName {
  param ([string]$Name)
  if (-not $Name) { return "hive_config_script" }
  return ([regex]::Replace($Name, '[^A-Za-z0-9_.-]', '_'))
}

function Get-SameSubnetAddress {
  param([string]$RemoteIP)

  $configs = @()
  try {
    $configs = Get-NetIPConfiguration | Where-Object { $_.IPv4Address }
  } catch {
    return $null
  }

  $remoteParts = $RemoteIP.Split('.')
  $hasRemoteV4 = $remoteParts.Length -eq 4
  $candidate16 = $null
  $fallback = $null

  foreach ($cfg in $configs) {
    foreach ($addr in $cfg.IPv4Address) {
      $ip = $addr.IPAddress
      if (-not $ip) { continue }
      if ($ip -like '127.*' -or $ip -like '169.254.*') { continue }
      if (-not $fallback) { $fallback = $ip }

      if (-not $hasRemoteV4) { continue }
      $parts = $ip.Split('.')
      if ($parts.Length -ne 4) { continue }

      if ("$($parts[0]).$($parts[1]).$($parts[2])" -eq "$($remoteParts[0]).$($remoteParts[1]).$($remoteParts[2])") {
        return $ip
      }
      if (-not $candidate16 -and "$($parts[0]).$($parts[1])" -eq "$($remoteParts[0]).$($remoteParts[1])") {
        $candidate16 = $ip
      }
    }
  }

  if ($candidate16) { return $candidate16 }
  return $fallback
}

function Get-ControllerHostAddress {
  param(
    [string]$RemoteIP,
    [int]$RemotePort
  )

  if ($env:CONFIG_HOST) { return $env:CONFIG_HOST }

  $candidate = $null
  try {
    $tnc = Test-NetConnection -ComputerName $RemoteIP -Port $RemotePort -WarningAction SilentlyContinue
    if ($tnc -and $tnc.SourceAddress -and $tnc.SourceAddress -notlike '127.*') {
      $candidate = $tnc.SourceAddress.ToString()
    }
  } catch { }

  if (-not $candidate) {
    $candidate = Get-SameSubnetAddress -RemoteIP $RemoteIP
  }

  if (-not $candidate) {
    try {
      $cfg = Get-NetIPConfiguration | Where-Object { $_.IPv4Address } | Select-Object -First 1
      if ($cfg -and $cfg.IPv4Address) {
        $candidate = $cfg.IPv4Address[0].IPAddress
      }
    } catch { }
  }

  return $candidate
}

function ConvertTo-ArgumentString {
  param([string[]]$Arguments)
  $escaped = $Arguments | ForEach-Object {
    if ($_ -match '[\s"]') {
      '"' + ($_ -replace '"','\"') + '"'
    } else {
      $_
    }
  }
  return [string]::Join(' ', $escaped)
}

function Invoke-StreamingProcess {
  param(
    [string]$FileName,
    [string[]]$Arguments,
    [string]$OutputPath
  )

  $psi = New-Object System.Diagnostics.ProcessStartInfo
  $psi.FileName = $FileName
  $psi.Arguments = ConvertTo-ArgumentString -Arguments $Arguments
  $psi.RedirectStandardOutput = $true
  $psi.RedirectStandardError = $true
  $psi.UseShellExecute = $false
  $psi.CreateNoWindow = $true

  $proc = New-Object System.Diagnostics.Process
  $proc.StartInfo = $psi
  $proc.Start() | Out-Null

  $writer = New-Object System.IO.StreamWriter($OutputPath, $false, [System.Text.Encoding]::UTF8)
  try {
    while (-not $proc.HasExited -or -not $proc.StandardOutput.EndOfStream -or -not $proc.StandardError.EndOfStream) {
      while (-not $proc.StandardOutput.EndOfStream) {
        $line = $proc.StandardOutput.ReadLine()
        if ($null -ne $line) {
          Write-Host $line
          $writer.WriteLine($line)
        }
      }
      while (-not $proc.StandardError.EndOfStream) {
        $line = $proc.StandardError.ReadLine()
        if ($null -ne $line) {
          Write-Host $line -ForegroundColor Yellow
          $writer.WriteLine($line)
        }
      }
      if (-not $proc.HasExited) { Start-Sleep -Milliseconds 50 }
    }
    $proc.WaitForExit()
    return $proc.ExitCode
  }
  finally {
    $writer.Dispose()
  }
}

$defaultCompat = @("-oHostKeyAlgorithms=+ssh-rsa", "-oPubkeyAcceptedAlgorithms=+ssh-rsa")
$SshOpts = Get-OptionArray -Value $env:SSH_OPTS -Default $defaultCompat
$ScpOpts = Get-OptionArray -Value $env:SCP_OPTS -Default $defaultCompat

$scriptPath = Join-Path (Get-Location) "scripts"
if (!(Test-Path $scriptPath)) {
  Write-Host "ERROR: scripts folder not found: $scriptPath" -ForegroundColor Red
  exit 1
}

$scripts = Get-ChildItem -Path $scriptPath -File | Sort-Object LastWriteTime -Descending
if ($scripts.Count -eq 0) {
  Write-Host "No scripts found in $scriptPath" -ForegroundColor Yellow
  exit 0
}

if ($MaxScripts -lt $scripts.Count) {
  $scripts = $scripts | Select-Object -First $MaxScripts
}

Write-Host "==> Selected:" -ForegroundColor Cyan
foreach ($s in $scripts) {
  Write-Host ("   - {0}" -f $s.Name)
}

$ConfigHost = Get-ControllerHostAddress -RemoteIP $RemoteIP -RemotePort $RemotePort
if ($ConfigHost) {
  Write-Host "==> Controller host detected as $ConfigHost" -ForegroundColor Green
} else {
  Write-Host "==> Controller host could not be auto-detected; relying on SSH metadata." -ForegroundColor Yellow
}

$timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
$resultsRoot = Join-Path (Get-Location) "remote_run_results"
if (!(Test-Path $resultsRoot)) { New-Item -ItemType Directory -Force -Path $resultsRoot | Out-Null }
$runDir = Join-Path $resultsRoot ("remote_run_results_{0}" -f $timestamp)
New-Item -ItemType Directory -Force -Path $runDir | Out-Null

$remoteEnv = "env HIVE_OFFLINE=1"
if ($ConfigHost) { $remoteEnv += " CONFIG_HOST=$ConfigHost HIVE_CONTROLLER_HOST=$ConfigHost" }

$runResults = @()

foreach ($script in $scripts) {
  Write-Host ("==> Running {0}" -f $script.Name) -ForegroundColor Cyan
  $safeName = Get-SafeName -Name $script.Name
  $remotePath = "$RemoteDir/$safeName"
  $localOutFile = Join-Path $runDir ("{0}.out" -f $script.Name)
  $localExitFile = Join-Path $runDir ("{0}.exit" -f $script.Name)

  $scpArgs = @()
  $scpArgs += $ScpOpts
  $scpArgs += "-O"
  $scpArgs += "-P"
  $scpArgs += $RemotePort
  $scpArgs += $script.FullName
  $scpArgs += "$RemoteUser@$RemoteIP:$remotePath"
  & scp @scpArgs

  $remoteCmd = "chmod +x '$remotePath'; if command -v timeout >/dev/null 2>&1; then $remoteEnv timeout $TimeoutSecs '$remotePath'; else $remoteEnv '$remotePath'; fi"
  $sshArgs = @()
  $sshArgs += $SshOpts
  $sshArgs += "-p"
  $sshArgs += $RemotePort
  $sshArgs += "$RemoteUser@$RemoteIP"
  $sshArgs += $remoteCmd
  $exitCode = Invoke-StreamingProcess -FileName "ssh" -Arguments $sshArgs -OutputPath $localOutFile
  Set-Content -Path $localExitFile -Value $exitCode
  $runResults += @{ Name = $script.Name; ExitCode = $exitCode; Log = $localOutFile }

  if ($Cleanup -eq "yes") {
    & ssh @SshOpts -p $RemotePort "$RemoteUser@$RemoteIP" "rm -f '$remotePath'" | Out-Null
  }
}

Write-Host "Logs stored in $runDir." -ForegroundColor Green
foreach ($result in $runResults) {
  Write-Host (" - {0}: exit code {1}" -f $result.Name, $result.ExitCode)
}
