#!/usr/bin/env bash
# run_hive_config.sh
# Looks in ./scripts/ (relative to PWD), picks candidate scripts (non-recursive),
# sorts them by mtime (newest first), and runs only the latest one by default.

set -u
set -o pipefail

REMOTE_USER="${REMOTE_USER:-root}"
REMOTE_PORT="${REMOTE_PORT:-22}"
REMOTE_DIR="${REMOTE_DIR:-/tmp}"
CLEANUP="${CLEANUP:-yes}"
SSH_OPTS="${SSH_OPTS:-}"
SCP_OPTS="${SCP_OPTS:-}"
TIMEOUT_SECS="${TIMEOUT_SECS:-900}"
MAX_SCRIPTS="${MAX_SCRIPTS:-1}"
SSH_NONINTERACTIVE_OPTS="-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null"
CONFIG_HOST="${CONFIG_HOST:-}"
if [ -z "$SSH_OPTS" ]; then
  SSH_OPTS="-oHostKeyAlgorithms=+ssh-rsa -oPubkeyAcceptedAlgorithms=+ssh-rsa"
fi
if [ -z "$SCP_OPTS" ]; then
  SCP_OPTS="-oHostKeyAlgorithms=+ssh-rsa -oPubkeyAcceptedAlgorithms=+ssh-rsa"
fi

if [ $# -ne 1 ]; then
  echo "Usage: $0 <remote-ip>"
  exit 2
fi
REMOTE_IP="$1"

clean_name() {
  local input="$1"
  input="${input// /_}"
  input="${input//(/_}"
  input="${input//)/_}"
  printf '%s' "${input//[^A-Za-z0-9_.-]/_}"
}

detect_controller_ip() {
  local remote_ip="$1"
  local candidate=""

  if command -v ip >/dev/null 2>&1; then
    candidate="$(ip route get "$remote_ip" 2>/dev/null | awk '/src/ { for (i=1;i<=NF;i++) if ($i=="src") { print $(i+1); exit } }')"
    if [ -n "$candidate" ]; then
      echo "$candidate"
      return
    fi
  fi

  if command -v python3 >/dev/null 2>&1 && command -v ip >/dev/null 2>&1; then
    candidate="$(python3 - "$remote_ip" <<'PY'
import ipaddress, json, subprocess, sys
try:
    remote = ipaddress.ip_address(sys.argv[1])
except Exception:
    sys.exit(0)
try:
    res = subprocess.run(["ip", "-json", "addr", "show"], capture_output=True, text=True, check=True)
except Exception:
    sys.exit(0)
try:
    data = json.loads(res.stdout)
except Exception:
    sys.exit(0)
for iface in data:
    for addr in iface.get("addr_info", []):
        if addr.get("family") != "inet":
            continue
        local = ipaddress.ip_address(addr.get("local"))
        prefix = int(addr.get("prefixlen", 32))
        network = ipaddress.ip_network(f"{local}/{prefix}", strict=False)
        if remote in network:
            print(str(local))
            sys.exit(0)
PY
)"
    if [ -n "$candidate" ]; then
      echo "$candidate"
      return
    fi
  fi

  if command -v hostname >/dev/null 2>&1; then
    candidate="$(hostname -I 2>/dev/null | awk '{print $1; exit}')"
    if [ -n "$candidate" ]; then
      echo "$candidate"
      return
    fi
  fi
}

SCRIPT_DIR="./scripts"
if [ ! -d "$SCRIPT_DIR" ]; then
  echo "ERROR: script directory not found: $SCRIPT_DIR"
  exit 3
fi

CANDIDATES=()
while IFS= read -r -d '' file; do
  name="${file##*/}"
  case "$name" in
    *.sh|*.bash|*.py)
      CANDIDATES+=("$file")
      ;;
    *)
      if [ -x "$file" ]; then
        CANDIDATES+=("$file")
      fi
      ;;
  esac
done < <(find "$SCRIPT_DIR" -mindepth 1 -maxdepth 1 -type f ! -name ".*" -print0)

TOTAL_FOUND=${#CANDIDATES[@]}
echo "==> Found $TOTAL_FOUND candidate(s) in $SCRIPT_DIR."
if [ "${#CANDIDATES[@]}" -eq 0 ]; then
  echo "Nothing to run. Exiting."
  exit 0
fi

mapfile -t SORTED < <(ls -t1 -- "${CANDIDATES[@]}")
CANDIDATES=("${SORTED[@]}")

if ! printf '%s\n' "$MAX_SCRIPTS" | grep -Eq '^[0-9]+$'; then MAX_SCRIPTS=1; fi
if [ "$MAX_SCRIPTS" -lt "${#CANDIDATES[@]}" ]; then
  CANDIDATES=("${CANDIDATES[@]:0:$MAX_SCRIPTS}")
fi

if [ -z "$CONFIG_HOST" ]; then
  CONFIG_HOST="$(detect_controller_ip "$REMOTE_IP")"
fi
if [ -n "$CONFIG_HOST" ]; then
  echo "==> Controller host detected as $CONFIG_HOST"
else
  echo "==> Controller host could not be auto-detected; relying on SSH session metadata."
fi

echo "==> Selected:"
for s in "${CANDIDATES[@]}"; do echo "   - $s"; done

TS="$(date +%Y%m%d_%H%M%S)"
RESULTS_ROOT="$(pwd)/remote_run_results"
mkdir -p "$RESULTS_ROOT"
LOCAL_RESULT_DIR="${RESULTS_ROOT}/remote_run_results_${TS}"
mkdir -p "$LOCAL_RESULT_DIR"

ssh_cmd() { ssh ${SSH_OPTS} ${SSH_NONINTERACTIVE_OPTS} -p "$REMOTE_PORT" "${REMOTE_USER}@${REMOTE_IP}" -- "$@"; }
scp_to_remote() { scp ${SCP_OPTS} -O -P "$REMOTE_PORT" "$1" "${REMOTE_USER}@${REMOTE_IP}:$2"; }

REMOTE_ENV_PREFIX="env HIVE_OFFLINE=1"
if [ -n "$CONFIG_HOST" ]; then
  REMOTE_ENV_PREFIX="$REMOTE_ENV_PREFIX CONFIG_HOST=$CONFIG_HOST HIVE_CONTROLLER_HOST=$CONFIG_HOST"
fi

for LOCAL_SCRIPT in "${CANDIDATES[@]}"; do
  BASENAME="$(basename "$LOCAL_SCRIPT")"
  SAFE_BASENAME="$(clean_name "$BASENAME")"
  if [ -z "$SAFE_BASENAME" ]; then
    SAFE_BASENAME="hive_config_script"
  fi
  REMOTE_SCRIPT_PATH="${REMOTE_DIR%/}/${SAFE_BASENAME}"
  LOCAL_OUT_FILE="${LOCAL_RESULT_DIR}/${BASENAME}.out"
  LOCAL_EXIT_FILE="${LOCAL_RESULT_DIR}/${BASENAME}.exit"
  echo "Running $LOCAL_SCRIPT on $REMOTE_IP..."
  scp_to_remote "$LOCAL_SCRIPT" "$REMOTE_SCRIPT_PATH"
  RUN_REMOTE="chmod +x '$REMOTE_SCRIPT_PATH'; if command -v timeout >/dev/null 2>&1; then ${REMOTE_ENV_PREFIX} timeout ${TIMEOUT_SECS}s '$REMOTE_SCRIPT_PATH'; else ${REMOTE_ENV_PREFIX} '$REMOTE_SCRIPT_PATH'; fi"
  {
    ssh_cmd "$RUN_REMOTE"
  } 2>&1 | tee "$LOCAL_OUT_FILE"
  SSH_STATUS=${PIPESTATUS[0]}
  echo "$SSH_STATUS" > "$LOCAL_EXIT_FILE"
  echo "Result stored in $LOCAL_OUT_FILE (exit code $SSH_STATUS)"
  if [ "$CLEANUP" = "yes" ]; then ssh_cmd "rm -f '$REMOTE_SCRIPT_PATH'" >/dev/null 2>&1; fi
done

echo "Logs available under $LOCAL_RESULT_DIR"
